#include <stdio.h>   
#include <unistd.h>  
#include <sys/stat.h>
#include <sys/dir.h>

/*
 * This file collects the common functions
 * shared by server's and client's programs.
 */

#include "defs.h"
#define DEBUG

extern const char *PRGNAME;

int
select_dir(struct direct *dir)
{
	struct stat file;

	/*
	 * Selecting function for scandir().
	 * Only regular files and dir are allowed.
	 */
	if (stat(dir->d_name, &file) == -1)
	{
		perror("stating in select_dir");
		return FALSE;
	}
	if (S_ISDIR(file.st_mode))
	{
		if (strcmp(dir->d_name, ".") != 0
				&& strcmp(dir->d_name, "..") != 0)
			return TRUE;
	}
	if (S_ISREG(file.st_mode))
		return TRUE;
	return FALSE;
}

int
readmsg(const int sock, char *msg, const size_t count)
{
	int rval;
	
	/*
	 * Reads a message msg of count byte from socket sock
	 * and prints it on stdout if DEBUG is enabled.
	 */
#ifdef DEBUG
	printf("%s: Rcvd:", PRGNAME);
#endif
	while ((rval = read(sock, msg, count)) <= 0)
		if ((rval = read(sock, msg, count)) == -1)
			perror("reading message from socket");
#ifdef DEBUG
	printf(" %-15s (%3u bytes)\n", msg, rval);
#endif
	return rval;
}

int
writemsg(const int sock, const char *msg, const size_t count)
{
	int rval;

	/*
	 * Writes message msg of count byte from socket sock
	 * and prints it on stdout if DEBUG is enabled.
	 */
#ifdef DEBUG
	printf("%s: Send:", PRGNAME);
#endif
	if ((rval = write(sock, msg, count)) == -1)
		perror("writing message to socket");
#ifdef DEBUG
	printf(" %-15s (%3u bytes)\n", msg, rval);
#endif
	return rval;
}

int
readstat(const int sock, struct stat *filestat)
{
	int rval;
	
	/*
	 * Reads stat informations from socket sock
	 * and prints a log message on stdout if DEBUG is enabled.
	 */
#ifdef DEBUG
	printf("%s: Rcvd:", PRGNAME);
#endif
	if ((rval = read(sock, filestat, sizeof(*filestat))) == -1)
	{
		perror("receiving file stat");
		exit(3);
	}
#ifdef DEBUG
	printf(" %-15s (%3u bytes)\n", "File stat", rval);
#endif
	return rval;
}

int
writestat(const int sock, const struct stat *filestat)
{
	int rval;
	
	/*
	 * Writes stat informations to socket sock
	 * and prints a log message on stdout if DEBUG is enabled.
	 */
#ifdef DEBUG
	printf("%s: Send:", PRGNAME);
#endif
	if ((rval = write(sock, filestat, sizeof(*filestat))) == -1)
	{
		perror("Sending file stat");
		exit(3);
	}
#ifdef DEBUG
	printf(" %-15s (%3u bytes)\n", "File stat", rval);
	sleep(2);
#endif
	return rval;
}
